/**
    \file CSED.h
    Header file for CSED distance transform class which, given
    and input binary image, calculates the corresponding distance transform.

    \author George J. Grevera, Ph.D., ggrevera@sju.edu

    Copyright (C) 2002, George J. Grevera

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
    USA or from http://www.gnu.org/licenses/gpl.txt.

    This General Public License does not permit incorporating this
    code into proprietary programs.  (So a hypothetical company such
    as GH (Generally Hectic) should NOT incorporate this code into
    their proprietary programs.)
 */
#ifndef CSED_h
#define CSED_h

#include "DistanceTransform.h"
#include <iostream>
using namespace std;
//----------------------------------------------------------------------
/** \brief Ragnemalm's CSED distance transform algorithm.
 *
 *  As described in I. Ragnemalm, "Neighborhoods for distance
 *  transformations using ordered propagation," CVGIP:IU 56(3):399-409,
 *  1992.
 *
 *  To produce code for the original algorithm, define the symbol ORIGINAL
 *  in the CSED .h or .cpp file and recompile.  In some cases this code
 *  exhibits errors (in the form of unassigned distance transform values for
 *  some points).  To produce an improved version, do not define this symbol.
 */
class CSED : public DistanceTransform {

public:
    CSED ( const int xSize, const int ySize, const bool unload=true )
        : DistanceTransform(xSize, ySize, unload)
    {
    }

    ~CSED ( ) {
        if (this->v!=NULL) {
            free( this->v );
            this->v = NULL;
        }
    }

    void doTransform ( const unsigned char* const I );

    /**
     * @param x is the x location
     * @param y is the y location
     * @param px is the parent's x location to be returned
     * @param py is the parent's y location to be returned
     * @return true if the parent is know; otherwise false.
     */
    virtual inline bool getP ( const int x, const int y, int& px, int& py )
    const {
        if (v==NULL) {
            px = py = -1;
            return false;
        }
        const int s = sub(x,y);
        px = x - v[s].x;
        py = y - v[s].y;
        return true;
    }

protected:
    P*  v;

private:
    inline void test ( const int px, const int py, const int ox, const int oy,
                       vector<P*>* list2, const int i );

    /// returns the length of a vector
    static inline double D ( const int x, const int y ) {
        return sqrt( (double)x * x + y * y );
    }

    /// returns the length of a vector
    static inline double D ( P& p ) {
        return sqrt( (double)p.x * p.x + p.y * p.y );
    }

    /// return the sign of i
    static inline int sgn ( int i ) {
        if (i<0)    return -1;
        if (i>0)    return  1;
        return 0;
    }
};

#endif
//----------------------------------------------------------------------
